#include <gint/display.h>
#include <gint/display-fx.h>
#include <gint/keyboard.h>
#include <gint/clock.h>
#include <gint/gint.h>
#include <dirent.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "../include/list.h"
#include <math.h>
#include <unistd.h>
#include <fcntl.h>

#define SCREEN 100
#define SCALE 60
#define DEPTH 6.0f
#define SCREEN_W 128
#define SCREEN_H 64


DIR *opendir_world_switch(char const *path) {
	return (DIR *)gint_world_switch(GINT_CALL(opendir, path));
}

bool strendswith(char const *str, char const *suffix)
{
    size_t suffix_len = strlen(suffix);
    size_t str_len = strlen(str);
    return str_len >= suffix_len && !strcmp(str + str_len - suffix_len, suffix);
}

list_t* get_ve_files(const char* path) {
    list_t* ve_list = malloc(sizeof *ve_list);
    List_Init(ve_list, SCREEN);

    DIR *dp = opendir(path);
	struct dirent *ent;

	while((ent = readdir(dp)) != NULL) {
        if(strendswith(ent->d_name, ".ve"))
    		List_Push(ve_list, ent->d_name);
	}

	closedir(dp);
    return ve_list;
}


typedef struct {
    float x, y, z;
} vec3;

typedef struct {
    int a, b;
} edge;

typedef struct {
    vec3* vertices;
    edge* edges;
    int num_vertices;
    int num_edges;
} model_t;

// هيكل لتمرير بيانات قراءة الملف داخل world switch
typedef struct {
    const char* path;
    char* buffer;
    int file_size;
    int error;
} read_file_data_t;

void read_file_callback(read_file_data_t* data) {
    int fd = open(data->path, O_RDONLY);
    if (fd < 0) {
        data->error = fd;
        return;
    }
    data->file_size = lseek(fd, 0, SEEK_END);
    lseek(fd, 0, SEEK_SET);

    data->buffer = (char*)malloc(data->file_size + 1);
    if (data->buffer == NULL) {
        data->error = -1;  // خطأ تخصيص ذاكرة
        close(fd);
        return;
    }
    int read_size = read(fd, data->buffer, data->file_size);
    if (read_size != data->file_size) {
        data->error = -1;
        free(data->buffer);
        data->buffer = NULL;
        close(fd);
        return;
    }
    data->buffer[data->file_size] = '\0';
    close(fd);
    data->error = 0;
}



void load_ve(const char* folder, const char* file, model_t* model) {
    if (!model) return;

    char full_path[150];
    strcpy(full_path, folder);
    strcat(full_path, "/");
    strcat(full_path, file);

    read_file_data_t data = { full_path, NULL, 0, 0 };
    gint_world_switch(GINT_CALL(read_file_callback, (void*)&data));

    if (data.error < 0 || data.buffer == NULL) {
        dclear(C_WHITE);
        dtext(1, 1, C_BLACK, "Error opening file");
        dupdate();
        return;
    }

    /* إذا كان model يحتوي سابقًا على بيانات حرّرها لتفادي تسرب الذاكرة */
    if (model->vertices) { free(model->vertices); model->vertices = NULL; }
    if (model->edges)    { free(model->edges);    model->edges    = NULL; }

    char *buffer = data.buffer;

    /* ----- تمريرة عدّ أولية على نسخة مؤقتة ----- */
    char *tmp = strdup(buffer);
    if (!tmp) {
        dclear(C_WHITE);
        dtext(1, 1, C_BLACK, "Memory error");
        dupdate();
        free(buffer);
        return;
    }
    /* إزالة أو استبدال \r لتفادي مشكلات strtok */
    for (char *p = tmp; *p; ++p) if (*p == '\r') *p = ' ';

    int count_v = 0, count_e = 0;
    char mode = 0;
    char *token = strtok(tmp, "\n");
    while (token != NULL) {
        if (token[0] == '#' || token[0] == '\0') { token = strtok(NULL, "\n"); continue; }
        if (strcmp(token, "V") == 0) { mode = 'V'; token = strtok(NULL, "\n"); continue; }
        if (strcmp(token, "E") == 0) { mode = 'E'; token = strtok(NULL, "\n"); continue; }
        if (mode == 'V') count_v++;
        else if (mode == 'E') count_e++;
        token = strtok(NULL, "\n");
    }
    free(tmp);

    /* حجز الذاكرة بناءً على العد */
    model->num_vertices = count_v;
    model->num_edges    = count_e;
    model->vertices = NULL;
    model->edges    = NULL;

    if (count_v > 0) model->vertices = (vec3*)malloc(count_v * sizeof(vec3));
    if (count_e > 0) model->edges    = (edge*) malloc(count_e  * sizeof(edge));

    if ((count_v > 0 && model->vertices == NULL) || (count_e > 0 && model->edges == NULL)) {
        if (model->vertices) free(model->vertices);
        if (model->edges)    free(model->edges);
        model->vertices = NULL;
        model->edges = NULL;
        model->num_vertices = model->num_edges = 0;

        dclear(C_WHITE);
        dtext(1, 1, C_BLACK, "Allocation failed");
        dupdate();
        free(buffer);
        return;
    }

    /* ----- تمريرة تحليل ثانية على نسخة أخرى ----- */
    char *buf_copy = strdup(buffer);
    if (!buf_copy) {
        if (model->vertices) free(model->vertices);
        if (model->edges)    free(model->edges);
        model->vertices = NULL;
        model->edges = NULL;
        model->num_vertices = model->num_edges = 0;

        dclear(C_WHITE);
        dtext(1, 1, C_BLACK, "Memory error");
        dupdate();
        free(buffer);
        return;
    }
    for (char *p = buf_copy; *p; ++p) if (*p == '\r') *p = ' ';

    mode = 0;
    int idx_v = 0, idx_e = 0;
    token = strtok(buf_copy, "\n");
    while (token != NULL) {
        if (token[0] == '#' || token[0] == '\0') { token = strtok(NULL, "\n"); continue; }
        if (strcmp(token, "V") == 0) { mode = 'V'; token = strtok(NULL, "\n"); continue; }
        if (strcmp(token, "E") == 0) { mode = 'E'; token = strtok(NULL, "\n"); continue; }

        if (mode == 'V') {
            float x, y, z;
            int got = sscanf(token, "%f %f %f", &x, &y, &z);
            if (got == 3 && idx_v < model->num_vertices) {
                model->vertices[idx_v].x = x;
                model->vertices[idx_v].y = y;
                model->vertices[idx_v].z = z;
                idx_v++;
            }
        } else if (mode == 'E') {
            int a, b;
            int got = sscanf(token, "%d %d", &a, &b);
            if (got == 2 && idx_e < model->num_edges) {
                a--; b--; /* تحويل من 1-based إلى 0-based */
                if (a < 0) a = 0;
                if (b < 0) b = 0;
                if (a >= model->num_vertices) a = model->num_vertices - 1;
                if (b >= model->num_vertices) b = model->num_vertices - 1;
                model->edges[idx_e].a = a;
                model->edges[idx_e].b = b;
                idx_e++;
            }
        }
        token = strtok(NULL, "\n");
    }

    free(buf_copy);
    free(buffer); /* تحرير البافر الذي ملأته دالة القراءة */
}





vec3 rotate(vec3 v, float ax, float ay) {
    float x1 = v.x * cosf(ay) + v.z * sinf(ay);
    float z1 = -v.x * sinf(ay) + v.z * cosf(ay);
    float y1 = v.y;

    float y2 = y1 * cosf(ax) - z1 * sinf(ax);
    float z2 = y1 * sinf(ax) + z1 * cosf(ax);

    vec3 res = {x1, y2, z2};
    return res;
}

void project(vec3 v, int* sx, int* sy) {
    float z = v.z + DEPTH;
    *sx = (int)((v.x / z) * SCALE + SCREEN_W / 2);
    *sy = (int)((v.y / z) * SCALE + SCREEN_H / 2);
}

int main(void)
{
    char const *folder = "/";

	dclear(C_WHITE);
	dupdate();
	list_t* ve_files = get_ve_files(folder);
	if (ve_files->count == 0) {
		dtext(1, 1, C_BLACK, "No .ve files found");
		dupdate();
		getkey();
		List_Destroy(ve_files);
		return 1;
	}
	int current_index = 0;
	model_t model;

	load_ve(folder, List_Get(ve_files, current_index), &model);

	dtext(1, 1, C_BLACK, List_Get(ve_files, current_index));
    dupdate();

    float ax = 0.0f, ay = 0.0f;

    while (1) {
        key_event_t key = getkey_opt(GETKEY_NONE, NULL);

        if (key.key == KEY_EXIT) break;

        if (key.key == KEY_LEFT) ay -= 0.05f;
        if (key.key == KEY_RIGHT) ay += 0.05f;
        if (key.key == KEY_UP) ax -= 0.05f;
        if (key.key == KEY_DOWN) ax += 0.05f;
		if (key.key == KEY_ARROW) {
			free(model.vertices);
			free(model.edges);
			current_index = (current_index + 1) % ve_files->count;
            load_ve(folder, List_Get(ve_files, current_index), &model);
			dtext(1, 1, C_BLACK, List_Get(ve_files, current_index));
			ax = 0.0f;
			ay = 0.0f;
		}

        dclear(C_WHITE);

        for (int i = 0; i < model.num_edges; i++) {
            vec3 va = rotate(model.vertices[model.edges[i].a], ax, ay);
            vec3 vb = rotate(model.vertices[model.edges[i].b], ax, ay);
            int x1, y1, x2, y2;
            project(va, &x1, &y1);
            project(vb, &x2, &y2);
            dline(x1, y1, x2, y2, C_BLACK);
        }

        dupdate();
        sleep_us_spin(33333);  // Approx 30 FPS
    }

    free(model.vertices);
    free(model.edges);
	for (size_t i = 0; i < ve_files->count; i++) {
		free(List_Get(ve_files, i));
	}
	List_Destroy(ve_files);

	return 1;
}
